<?php
/**
 * Security Test & Integration Helper
 * 
 * This script helps test and validate the security fixes
 * to ensure Bricks Builder and other page builders work correctly
 * after implementing the security improvements.
 * 
 * @package JenvaBackupMigration
 * @since 1.0.1
 */

// Only run in WordPress admin or CLI environment
if (!defined('ABSPATH') && !defined('WP_CLI')) {
    exit('This script must be run within WordPress environment');
}

class JBM_SecurityTestHelper {
    
    /**
     * Test serialized data handling
     */
    public static function testSerializedDataHandling(): array {
        $handler = new \JenvaBackupMigration\Restore\SerializedDataHandler();
        
        $test_cases = [
            // Valid serialized array with URL
            'valid_array' => serialize([
                'image_url' => 'https://oldsite.com/image.jpg',
                'text' => 'Some content with https://oldsite.com/page/',
                'nested' => [
                    'link' => 'https://oldsite.com/nested-page/'
                ]
            ]),
            
            // Corrupted serialized data (wrong byte count)
            'corrupted_string' => 's:25:"https://oldsite.com/image.jpg";', // Wrong: should be s:30
            
            // Complex Bricks-like structure
            'bricks_structure' => serialize([
                'elements' => [
                    [
                        'name' => 'image',
                        'settings' => [
                            'image' => ['url' => 'https://oldsite.com/bricks-image.jpg'],
                            'link' => 'https://oldsite.com/page/'
                        ]
                    ]
                ]
            ])
        ];
        
        $results = [];
        
        foreach ($test_cases as $name => $data) {
            echo "Testing: $name\n";
            
            $migrated = $handler->migrateUrls(
                $data, 
                'https://oldsite.com', 
                'https://newsite.com'
            );
            
            $results[$name] = [
                'original' => $data,
                'migrated' => $migrated,
                'valid_after' => @unserialize($migrated) !== false,
                'contains_new_url' => strpos($migrated, 'https://newsite.com') !== false,
                'contains_old_url' => strpos($migrated, 'https://oldsite.com') !== false
            ];
            
            echo "  - Valid after migration: " . ($results[$name]['valid_after'] ? 'YES' : 'NO') . "\n";
            echo "  - Contains new URL: " . ($results[$name]['contains_new_url'] ? 'YES' : 'NO') . "\n";
            echo "  - Still contains old URL: " . ($results[$name]['contains_old_url'] ? 'YES' : 'NO') . "\n";
            echo "\n";
        }
        
        $stats = $handler->getStats();
        echo "Handler Statistics:\n";
        echo "  - Processed: {$stats['processed']}\n";
        echo "  - Repaired: {$stats['repaired']}\n";
        echo "  - Failed: {$stats['failed']}\n";
        
        return $results;
    }
    
    /**
     * Test SQL injection protection in DatabaseImporter
     */
    public static function testSqlInjectionProtection(): array {
        $importer = new \JenvaBackupMigration\Restore\DatabaseImporter();
        
        $test_statements = [
            // Valid statements that should pass
            'valid_create' => "CREATE TABLE `wp_test` (`id` int(11) NOT NULL);",
            'valid_insert' => "INSERT INTO `wp_posts` (`post_title`) VALUES ('Test');",
            'valid_update' => "UPDATE `wp_options` SET `option_value` = 'test';",
            
            // Dangerous statements that should be blocked
            'dangerous_drop_db' => "DROP DATABASE wordpress;",
            'dangerous_load_file' => "SELECT LOAD_FILE('/etc/passwd');",
            'dangerous_outfile' => "SELECT * INTO OUTFILE '/tmp/dump.txt' FROM wp_users;",
            'dangerous_grant' => "GRANT ALL PRIVILEGES ON *.* TO 'hacker'@'%';",
            
            // Edge cases for restore operations
            'create_with_if_not_exists' => "CREATE TABLE IF NOT EXISTS `wp_custom` (`id` int);",
            'replace_into' => "REPLACE INTO `wp_test` VALUES (1, 'test');",
        ];
        
        $results = [];
        
        foreach ($test_statements as $name => $sql) {
            // Use reflection to test the private method
            $reflection = new ReflectionClass($importer);
            $method = $reflection->getMethod('isSafeStatement');
            $method->setAccessible(true);
            
            $is_safe = $method->invoke($importer, $sql);
            $should_be_safe = strpos($name, 'dangerous') === false;
            
            $results[$name] = [
                'sql' => $sql,
                'is_safe' => $is_safe,
                'should_be_safe' => $should_be_safe,
                'correct' => $is_safe === $should_be_safe
            ];
            
            $status = $results[$name]['correct'] ? 'PASS' : 'FAIL';
            echo "$name: $status (Safe: " . ($is_safe ? 'YES' : 'NO') . ")\n";
        }
        
        return $results;
    }
    
    /**
     * Test Bricks Builder integration
     */
    public static function testBricksIntegration(): array {
        if (!class_exists('\JenvaBackupMigration\Compatibility\BricksHandler')) {
            return ['error' => 'BricksHandler class not found'];
        }
        
        $handler = new \JenvaBackupMigration\Compatibility\BricksHandler();
        
        // Test if Bricks detection works
        $is_active = $handler->isBricksActive();
        
        echo "Bricks Builder Detection:\n";
        echo "  - Detected as active: " . ($is_active ? 'YES' : 'NO') . "\n";
        
        // If Bricks is detected, test URL migration
        $results = ['bricks_detected' => $is_active];
        
        if ($is_active) {
            echo "\nTesting Bricks URL Migration...\n";
            
            $migration_result = $handler->migrateUrls(
                'https://old-bricks-site.com',
                'https://new-bricks-site.com'
            );
            
            $results['migration'] = $migration_result;
            
            echo "Migration Results:\n";
            echo "  - Postmeta updated: {$migration_result['postmeta_updated']}\n";
            echo "  - Options updated: {$migration_result['options_updated']}\n";
            echo "  - Errors: " . count($migration_result['errors']) . "\n";
            
            if (!empty($migration_result['errors'])) {
                echo "\nErrors encountered:\n";
                foreach ($migration_result['errors'] as $error) {
                    echo "  - $error\n";
                }
            }
        }
        
        return $results;
    }
    
    /**
     * Comprehensive security and functionality test
     */
    public static function runAllTests(): array {
        echo "=== Jenva Backup Migration Security & Integration Tests ===\n\n";
        
        $all_results = [];
        
        // Test 1: Serialized Data Handling
        echo "1. SERIALIZED DATA HANDLING TEST\n";
        echo str_repeat('-', 50) . "\n";
        $all_results['serialized_data'] = self::testSerializedDataHandling();
        echo "\n";
        
        // Test 2: SQL Injection Protection
        echo "2. SQL INJECTION PROTECTION TEST\n";
        echo str_repeat('-', 50) . "\n";
        $all_results['sql_injection'] = self::testSqlInjectionProtection();
        echo "\n";
        
        // Test 3: Bricks Builder Integration
        echo "3. BRICKS BUILDER INTEGRATION TEST\n";
        echo str_repeat('-', 50) . "\n";
        $all_results['bricks_integration'] = self::testBricksIntegration();
        echo "\n";
        
        // Summary
        echo "=== TEST SUMMARY ===\n";
        $total_tests = 0;
        $passed_tests = 0;
        
        // Count serialized data test results
        foreach ($all_results['serialized_data'] as $result) {
            $total_tests++;
            if ($result['valid_after'] && $result['contains_new_url'] && !$result['contains_old_url']) {
                $passed_tests++;
            }
        }
        
        // Count SQL injection test results
        foreach ($all_results['sql_injection'] as $result) {
            $total_tests++;
            if ($result['correct']) {
                $passed_tests++;
            }
        }
        
        echo "Tests Passed: $passed_tests / $total_tests\n";
        
        if ($passed_tests === $total_tests) {
            echo "✅ All security fixes are working correctly!\n";
        } else {
            echo "⚠️  Some tests failed - please review the implementation.\n";
        }
        
        return $all_results;
    }
}

// Auto-run if accessed directly
if (!defined('WP_CLI') && isset($_GET['run_jbm_tests'])) {
    header('Content-Type: text/plain');
    JBM_SecurityTestHelper::runAllTests();
    exit;
}